package com.clobot.minibasic.data.update

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Environment
import androidx.core.content.FileProvider
import com.clobot.minibasic.BuildConfig
import com.clobot.minibasic.R
import kotlinx.coroutines.flow.flow
import java.io.File

@SuppressLint("StaticFieldLeak")
object UpdateManager {

    lateinit var context: Context

    fun init(context: Context) {
        this.context = context
    }


    sealed class CheckResult {
        class Found(val fileName: String, val version: String): CheckResult()
        data object NotFound: CheckResult()
        class Fail(val errorMessage: String): CheckResult()
    }

    fun check(hostName: String, ftpPort: Int, remoteDir: String, prefix: String, suffix: String) = flow {
        val dirDownloader = DirDownloader()
        dirDownloader.download(hostName, ftpPort, remoteDir, false).collect() {
            val checkResult = when(it) {
                is DirDownloader.Result.Success -> {
                    var highestFileName = ""
                    for (dirFtpFile in it.dirFtpFileArr) {
                        if (dirFtpFile.ftpFile.isFile) {
                            if (dirFtpFile.ftpFile.name.startsWith(prefix, true) &&
                                dirFtpFile.ftpFile.name.endsWith(suffix, true)) {

                                if (highestFileName.isEmpty() || dirFtpFile.ftpFile.name > highestFileName)
                                    highestFileName = dirFtpFile.ftpFile.name
                            }
                        }
                    }

                    if (highestFileName.isNotEmpty()) {
                        if (highestFileName.length == prefix.length + 8 + suffix.length) {
                            val fileVersion = StringBuilder()
                            fileVersion.append(highestFileName[prefix.length])
                            fileVersion.append(".")
                            fileVersion.append(highestFileName[prefix.length + 2])
                            fileVersion.append(highestFileName[prefix.length + 3])
                            fileVersion.append(".")
                            fileVersion.append(highestFileName[prefix.length + 5])
                            fileVersion.append(highestFileName[prefix.length + 6])
                            fileVersion.append(highestFileName[prefix.length + 7])

                            CheckResult.Found(highestFileName, fileVersion.toString())
                        }
                        else
                            CheckResult.Fail("\"$highestFileName\" file name version format error!")
                    } else
                        CheckResult.NotFound
                }
                is DirDownloader.Result.Fail -> CheckResult.Fail(it.message)
            }
            emit(checkResult)
        }
    }

    sealed class DownloadResult {
        class Downloading(val progressMessage: String): DownloadResult()
        data object Complete: DownloadResult()
        class Fail(val errorMessage: String): DownloadResult()
    }

    fun isValidRemoteFileName(remoteFileName: String): Boolean {
        return remoteFileName.isNotEmpty()
    }
    fun download(hostName: String, httpPort: Int, remoteDir: String, remoteFileName:String, apkFileName:String) = flow {
        FileDownloader().download("http://${hostName}", httpPort,
            "${remoteDir}/${remoteFileName}",
            "${Environment.getExternalStorageDirectory()}/${context.resources.getString(R.string.app_name)}/${apkFileName}"
        ).collect() {
            when(it) {
                is FileDownloader.Result.Begin -> {}
                is FileDownloader.Result.DownloadFile -> {
                    when(val subResult = it.subResult) {
                        is FileDownloader.Result.DownloadFile.Begin -> {}
                        is FileDownloader.Result.DownloadFile.Downloading ->
                            emit(DownloadResult.Downloading(subResult.progressMessage))
                        is FileDownloader.Result.DownloadFile.Success -> {}
                        is FileDownloader.Result.DownloadFile.Fail -> emit(DownloadResult.Fail(subResult.errorMessage))
                    }
                }
                is FileDownloader.Result.Complete -> {
                    emit(DownloadResult.Complete)
                }
                is FileDownloader.Result.Fail -> {
                    emit(DownloadResult.Fail(it.errorMessage))
                }
            }
        }
    }











    fun deleteFile(fileName: String): Boolean {
        return File("${Environment.getExternalStorageDirectory()}/${context.resources.getString(R.string.app_name)}/${fileName}").delete()
    }
    fun isFileExits(fileName: String): Boolean {
        return File("${Environment.getExternalStorageDirectory()}/${context.resources.getString(R.string.app_name)}/${fileName}").isFile
    }
    fun install(fileName: String) {
        val file = File("${Environment.getExternalStorageDirectory()}/${context.resources.getString(R.string.app_name)}/${fileName}")
        val fileUri : Uri = try {
            FileProvider.getUriForFile(context, "${BuildConfig.APPLICATION_ID}.provider", file)
        } catch (e: IllegalArgumentException) {
            null
        } ?: return

        val intent = Intent(Intent.ACTION_VIEW)
        intent.setDataAndType(fileUri, "application/vnd.android.package-archive")
        //intent.flags = Intent.FLAG_ACTIVITY_CLEAR_TOP or Intent.FLAG_ACTIVITY_NEW_TASK
        intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
        intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
        //intent.putExtra(Intent.EXTRA_NOT_UNKNOWN_SOURCE, true)
        try {
            context.startActivity(intent)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }




}