package com.clobot.minibasic.data.update

import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.FlowCollector
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import okhttp3.ResponseBody
import retrofit2.Response
import retrofit2.Retrofit
import retrofit2.converter.moshi.MoshiConverterFactory
import retrofit2.http.GET
import retrofit2.http.Streaming
import retrofit2.http.Url
import java.io.File
import java.text.DecimalFormat


class FileDownloader {

    // mainifest 에 android:usesCleartextTraffic="true" 필수

    interface DownloadFileService {
        @Streaming
        @GET
        suspend fun download(@Url fileUrl: String): Response<ResponseBody>
    }

    sealed class Result {
        object Begin : Result()
        class DownloadFile(val subResult: SubResult) : Result() {
            sealed class SubResult(val index: Int, val remotePath: String, val localPath: String)
            class Begin(index: Int, remotePath: String, localPath: String) : SubResult(index, remotePath, localPath)
            class Downloading(index: Int, remotePath: String, localPath: String, val progressMessage: String) : SubResult(index, remotePath, localPath)
            class Success(index: Int, remotePath: String, localPath: String) : SubResult(index, remotePath, localPath)
            class Fail(index: Int, remotePath: String, localPath: String, val errorMessage: String) : SubResult(index, remotePath, localPath)
        }
        class Complete(val successCount: Int) : Result()
        class Fail(val errorMessage: String) : Result()
    }

    private fun getDownloadService(hostName:String, port:Int): DownloadFileService {
        val retrofit = Retrofit.Builder()
            .baseUrl("$hostName:$port/")
            .addConverterFactory(MoshiConverterFactory.create())
            .build()

        return retrofit.create(DownloadFileService::class.java)
    }
    private suspend fun downloadFile(flow:FlowCollector<Result>, downloadFileService: DownloadFileService, index: Int, remotePath: String, localPath: String) : Boolean {

        flow.emit(Result.DownloadFile(Result.DownloadFile.Begin(index, remotePath, localPath)))

        val result = downloadFileService.download(remotePath)

        if (!result.isSuccessful) {
            flow.emit(
                Result.DownloadFile(
                    Result.DownloadFile.Fail(
                        index,
                        remotePath,
                        localPath,
                        "downloadFileService.downloadFile()"
                    )
                )
            )
            return false
        }

        val file = File(localPath)
        if (file.exists())
            file.delete()
        else
            file.parentFile?.mkdirs()

        val input = result.body()!!.byteStream()
        val fos = file.outputStream()

        fos.use { output ->
            val contentLength = result.body()!!.contentLength()
            var progressLength = 0L
            val buffer = ByteArray(DEFAULT_BUFFER_SIZE * 1024)
            //val buffer = ByteArray(128) // 테스트용
            var read: Int
            while (input.read(buffer).also { read = it } != -1) {
                output.write(buffer, 0, read)
                progressLength += read
                val level = progressLength * 100 / contentLength
                flow.emit(
                    Result.DownloadFile(
                        Result.DownloadFile.Downloading(
                            index, remotePath, localPath,
                            "${DecimalFormat("#,###").format(progressLength)} / ${
                                DecimalFormat("#,###").format(contentLength)
                            } byte(s) ($level%)"
                        )
                    )
                )
            }
            output.flush()
        }

        flow.emit(
            Result.DownloadFile(
                Result.DownloadFile.Success(
                    index,
                    remotePath,
                    localPath
                )
            )
        )
        return true
    }

    // host: ex) http://download.clobot.co.kr
    // port: ex) 8088
    // remoteDir: ex) aaa/bbb (o),   aaa/bbb/ (X)
    // remoteFileArr: ex) /c.txt, /ddd/eee/q.txt, f.txt (X)
    // localDir: ex) aaa, bbb/ccc, /ddd (X), ddd/(X)
    fun downloads(hostName:String, port:Int, remoteDir: String, remoteFileArr: Array<String>, localDir: String): Flow<Result> {
        return flow {
            try {
                emit(Result.Begin)

                val downloadFileService = getDownloadService(hostName, port)

                var successCount = 0
                for ((index, remoteFile) in remoteFileArr.withIndex()) {

                    val remotePath = "${remoteDir}${remoteFile}"
                    val localPath = "${localDir}${remoteFile}"

                    try {
                        if (downloadFile(this, downloadFileService, index, remotePath, localPath))
                            successCount++;
                    } catch (e: Exception) {
                        emit(
                            Result.DownloadFile(
                                Result.DownloadFile.Fail(
                                    index,
                                    remotePath,
                                    localPath,
                                    e.message!!
                                )
                            )
                        )
                    }
                }
                emit(Result.Complete(successCount))
            } catch (e: Exception) {
                emit(Result.Fail(e.message!!))
            }
        }.flowOn(Dispatchers.IO)
    }

    fun download(hostName:String, port:Int, remotePath: String, localPath: String): Flow<Result> {
        //val remoteDir = remotePath.substringBeforeLast("/")
        return flow {
            try {
                emit(Result.Begin)

                val downloadFileService = getDownloadService(hostName, port)

                var successCount = 0

                try {
                    if (downloadFile(this, downloadFileService, 0, remotePath, localPath))
                        successCount++
                } catch (e: Exception) {
                    emit(
                        Result.DownloadFile(
                            Result.DownloadFile.Fail(
                                0,
                                remotePath,
                                localPath,
                                e.message!!
                            )
                        )
                    )
                }
                emit(Result.Complete(successCount))

            } catch (e: Exception) {
                emit(Result.Fail(e.message!!))
            }
        }.flowOn(Dispatchers.IO)
    }
}
