package com.clobot.minibasic.data.task.mini.loop.move

import androidx.compose.runtime.Composable
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import com.clobot.minibasic.R
import com.clobot.minibasic.data.RobotManager
import com.clobot.minibasic.data.SetupManger
import com.clobot.minibasic.data.task.MiniImageBg
import com.clobot.minibasic.data.task.MiniTextImageButton
import com.clobot.minibasic.data.task.SceneTask
import com.clobot.minibasic.data.task.lpToSp
import com.clobot.minibasic.data.task.mini.SystemButton1
import com.clobot.minibasic.data.task.mini.SystemButton2
import com.clobot.minibasic.data.task.mini.SystemCompose
import com.clobot.minibasic.data.task.mini.SystemMainText
import com.clobot.minibasic.view.layer1_scene.SceneView
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.launch

class MoveNavigationShowFailSceneTask(
    private val destPoi: String,
    private val tryCount: Int,
    private val reason: MoveNavigationStartSceneTask.Result
) : SceneTask() {
    enum class Result {
        RETRY, /* STAY,*/ ABORT,
    }

    private val resultChannel = Channel<Result>()

    override val timeoutSec: Int = SetupManger.Open.Move.navigationShowFailSceneTimeoutSec
    override suspend fun onRun(): Result {
        return coroutineScope {
            val countJob = launch {
                runCountJob() {
                    updateView()
                }
                resultChannel.send(if (tryCount < SetupManger.Open.Move.navigationShowFailSceneTryCountMax - 1) Result.RETRY else Result.ABORT)
            }

            try {
                resultChannel.receive()
            } finally {
                countJob.cancel()
            }
        }
    }

    private fun updateView() {
        setSceneView(
            SceneView.MoveNavigationShowFail(
                destPoi,
                tryCount, SetupManger.Open.Move.navigationShowFailSceneTryCountMax, reason,
                countSec,
                { onRetry() },
                { onAbort() },
            )
        )
    }

    private fun onRetry() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.RETRY)
        }
    }

    private fun onAbort() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.ABORT)
        }
    }
}


@Composable
fun MoveNavigationShowFailSceneScreen(sceneView: SceneView.MoveNavigationShowFail) {
    MiniImageBg(R.drawable.scene_charge_show_fail_bg) {
        SystemMainText(text = buildAnnotatedString {

            withStyle(SpanStyle(SystemCompose.highlightTextColor)) {
                append(sceneView.destPoi)
            }
            append("(으)로 이동하는데 실패하였습니다.\n")

            withStyle(
                SpanStyle(fontSize = lpToSp(40))
            ) {
                append("(${sceneView.tryCount + 1}/${sceneView.tryCountMax})\n")
            }
            withStyle(
                SpanStyle(Color.Red, lpToSp(50))
            ) {
                val msg = when (val reason = sceneView.reason) {
                    is MoveNavigationStartSceneTask.Result.AlreadyInLocation -> "이미 위치함"
                    is MoveNavigationStartSceneTask.Result.NotExistPoi -> "위치가 존재하지 않음"
                    is MoveNavigationStartSceneTask.Result.UserAbort -> "사용자 취소"
                    is MoveNavigationStartSceneTask.Result.Timeout -> "시간 초과"
                    is MoveNavigationStartSceneTask.Result.StartNavigationResultResult -> {
                        when (reason.startNavigationResult) {
                            is RobotManager.StartNavigationResult.Result -> "Result: (${reason.startNavigationResult.result})"
                            is RobotManager.StartNavigationResult.OnResult -> "OnResult: (${reason.startNavigationResult.status}, ${reason.startNavigationResult.responseString})"
                            is RobotManager.StartNavigationResult.OnError -> "OnError: (${reason.startNavigationResult.errorCode}, ${reason.startNavigationResult.errorString})"
                        }
                    }

                }
                append(msg)
            }
        })

        SystemButton1(
            "재시도" + if (sceneView.tryCount < sceneView.tryCountMax - 1) "(${sceneView.countSec})" else "",
            sceneView.onRetry
        )
        SystemButton2(
            "취소" + if (sceneView.tryCount >= sceneView.tryCountMax - 1) "(${sceneView.countSec})" else "",
            sceneView.onAbort,
        )
    }
}
