package com.clobot.minibasic.data.task.mini.loop.move

import android.widget.VideoView
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.viewinterop.AndroidView
import com.clobot.minibasic.R
import com.clobot.minibasic.data.RobotManager
import com.clobot.minibasic.data.SetupManger
import com.clobot.minibasic.data.task.MiniCompose
import com.clobot.minibasic.data.task.SceneTask
import com.clobot.minibasic.data.task.lp
import com.clobot.minibasic.data.task.lpToSp
import com.clobot.minibasic.data.task.mini.SystemMainText
import com.clobot.minibasic.data.task.mini.SystemProgressBar
import com.clobot.minibasic.data.task.mini.SystemProgressText
import com.clobot.minibasic.data.task.mini.SystemSubText
import com.clobot.minibasic.view.layer1_scene.SceneView
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.launch
import kotlin.coroutines.cancellation.CancellationException

class MoveLeaveStartSceneTask(private val tryCount: Int) : SceneTask() {

    sealed class Result {
        data object Ok : Result()
        class LeaveChargeResult(val leaveChargePileResult: RobotManager.LeaveChargePileResult) :
            Result()

        data object UserAbort : Result()

        data object Timeout : Result()
    }

    override val timeoutSec: Int = SetupManger.Open.Move.leaveStartSceneTimeoutSec

    val resultChannel = Channel<Result>()
    override suspend fun onRun(): Result {
        if (isPause)
            return Result.UserAbort
        if (!RobotManager.isChargingSf.value!!)
            return Result.Ok

        return coroutineScope {

            val countJob = launch {
                runCountJob() {
                    updateView()
                }
                RobotManager.stopNavigation()
                resultChannel.send(Result.Timeout)
            }
            val leaveChargePileJob = launch {
                resultChannel.send(
                    Result.LeaveChargeResult(
                        RobotManager.leaveChargePile(
                            SetupManger.Open.Move.leaveChargePileSpeed,
                            SetupManger.Open.Move.leaveChargePileDistance
                        )
                    )
                )
            }
            try {
                resultChannel.receive()
            } catch (e: CancellationException) {
                Result.UserAbort
            } finally {
                countJob.cancel()
                leaveChargePileJob.cancel()
            }
        }
    }

    private fun updateView() {
        setSceneView(
            SceneView.MoveLeaveStart(
                tryCount,
                SetupManger.Open.Move.leaveShowFailSceneTryCountMax,
                countSec,
                { onPause() },
            )
        )
    }

    private fun onPause() {
        RobotManager.stopNavigation()
        taskCoroutineScope?.launch {
            resultChannel.send(Result.UserAbort)
        }
    }
    override fun onPause(isPause: Boolean) {
        if (isPause)
            onPause()
    }

}
@Composable
fun MoveLeaveStartSceneScreen(sceneView: SceneView.MoveLeaveStart) {
    Box(Modifier.lp(0, 0, MiniCompose.bg_width, MiniCompose.bg_height)) {
        val videoPath =
            "android.resource://${LocalContext.current.packageName}/${R.raw.move_mp4}"
        AndroidView(
            modifier = Modifier.fillMaxSize(),
            factory = { mContext ->
                VideoView(mContext).apply {
                    setVideoPath(videoPath)
                    setOnPreparedListener { mediaPlayer ->
                        mediaPlayer.isLooping = true
                    }
                    start()
                }
            }
        )
        SystemMainText(text = buildAnnotatedString {
            append("충전기에서 나오는 중입니다.")
            withStyle(
                SpanStyle(fontSize = lpToSp(40))
            ) {
                append("(${sceneView.tryCount + 1}/${sceneView.tryCountMax})\n")
            }
            append("잠시만 뒤로 물러서 주세요.")
        })

        SystemProgressBar {
            CircularProgressIndicator(modifier = Modifier.fillMaxSize(), color = Color.White)
        }
        SystemProgressText(sceneView.countSec)

        SystemSubText("화면을 터치하면 로봇을 멈출 수 있습니다.")
    }
    Box(
        modifier = Modifier
            .fillMaxSize()
            .clickable { sceneView.onPause() }
    ) {
    }

}
