package com.clobot.minibasic.data.task.mini.loop.move

import androidx.compose.runtime.Composable
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import com.clobot.minibasic.R
import com.clobot.minibasic.data.RobotManager
import com.clobot.minibasic.data.SetupManger
import com.clobot.minibasic.data.task.MiniImageBg
import com.clobot.minibasic.data.task.SceneTask
import com.clobot.minibasic.data.task.lpToSp
import com.clobot.minibasic.data.task.mini.SystemButton1
import com.clobot.minibasic.data.task.mini.SystemButton2
import com.clobot.minibasic.data.task.mini.SystemButton3
import com.clobot.minibasic.data.task.mini.SystemMainText
import com.clobot.minibasic.view.layer1_scene.SceneView
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.launch

class MoveLeaveShowFailSceneTask(private val tryCount: Int, private val reason: MoveLeaveStartSceneTask.Result) : SceneTask() {
    enum class Result {
        RETRY, STAY, ABORT,
    }
    private val resultChannel = Channel<Result>()

    override val timeoutSec: Int = SetupManger.Open.Move.leaveShowFailSceneTimeoutSec
    override suspend fun onRun(): Result {
        return coroutineScope {
            val countJob = launch {
                runCountJob() {
                    updateView()
                }
                resultChannel.send(if (tryCount < SetupManger.Open.Move.leaveShowFailSceneTryCountMax - 1) Result.RETRY else Result.ABORT)
            }

            try {
                resultChannel.receive()
            } finally {
                countJob.cancel()
            }
        }
    }
    private fun updateView() {
        setSceneView(
            SceneView.MoveLeaveShowFail(tryCount, SetupManger.Open.Move.leaveShowFailSceneTryCountMax, reason,
                countSec,
                { onRetry() },
                { onStay() },
                { onAbort() },
            )
        )
    }
    private fun onRetry() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.RETRY)
        }
    }
    private fun onStay() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.STAY)
        }
    }

    private fun onAbort() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.ABORT)
        }
    }
}



@Composable
fun MoveLeaveShowFailSceneScreen(sceneView: SceneView.MoveLeaveShowFail) {
    MiniImageBg(R.drawable.scene_charge_show_fail_bg) {
        SystemMainText(text = buildAnnotatedString {
            append("충전기에서 나오는데 실패하였습니다.")

            withStyle(
                SpanStyle(fontSize = lpToSp(40))
            ) {
                append("(${sceneView.tryCount + 1}/${sceneView.tryCountMax})\n")
            }
            withStyle(
                SpanStyle(Color.Red, lpToSp(50))
            ) {
                val msg = when (val reason = sceneView.reason) {
                    is MoveLeaveStartSceneTask.Result.Ok -> ""
                    is MoveLeaveStartSceneTask.Result.UserAbort -> "사용자 취소"
                    is MoveLeaveStartSceneTask.Result.Timeout -> "시간 초과"
                    is MoveLeaveStartSceneTask.Result.LeaveChargeResult -> {
                        when (reason.leaveChargePileResult) {
                            is RobotManager.LeaveChargePileResult.Result -> "Result: (${reason.leaveChargePileResult.result})"
                            is RobotManager.LeaveChargePileResult.OnResult -> "OnResult: (${reason.leaveChargePileResult.result}, ${reason.leaveChargePileResult.message})"
                        }
                    }

                }
                append(msg)
            }
        })

        SystemButton1(
            "재시도" + if (sceneView.tryCount < sceneView.tryCountMax - 1) "(${sceneView.countSec})" else "",
            sceneView.onRetry
        )
        SystemButton2(
            "대기" + if (sceneView.tryCount >= sceneView.tryCountMax - 1) "(${sceneView.countSec})" else "",
            sceneView.onStay,
        )
        SystemButton3(
            "취소",
            sceneView.onAbort,
        )
    }
}
