package com.clobot.minibasic.data.task.mini.loop.mode.guide.move

import androidx.compose.runtime.Composable
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import com.clobot.minibasic.R
import com.clobot.minibasic.data.RobotManager
import com.clobot.minibasic.data.SetupManger
import com.clobot.minibasic.data.task.MiniImageBg
import com.clobot.minibasic.data.task.MiniTextImageButton
import com.clobot.minibasic.data.task.SceneTask
import com.clobot.minibasic.data.task.lpToSp
import com.clobot.minibasic.data.task.mini.SystemCompose
import com.clobot.minibasic.data.task.mini.SystemMainText
import com.clobot.minibasic.data.task.mini.loop.move.MoveNavigationStartSceneTask
import com.clobot.minibasic.view.layer1_scene.SceneView
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.launch

class GuideMoveNavigationShowFailSceneTask(
    private val destPoi: String,
    private val tryCount: Int,
    private val reason: MoveNavigationStartSceneTask.Result
) : SceneTask() {
    enum class Result {
        RETRY, ABORT, CHANGE_DEST,
    }

    private val resultChannel = Channel<Result>()

    override val timeoutSec: Int = SetupManger.Open.Move.navigationShowFailSceneTimeoutSec
    override suspend fun onRun(): Result {
        return coroutineScope {
            val countJob = launch {
                runCountJob() {
                    updateView()
                }
                resultChannel.send(if (tryCount < SetupManger.Open.Move.navigationShowFailSceneTryCountMax - 1) Result.RETRY else Result.ABORT)
            }

            try {
                resultChannel.receive()
            } finally {
                countJob.cancel()
            }
        }
    }

    private fun updateView() {
        setSceneView(
            SceneView.GuideMoveNavigationShowFail(
                destPoi,
                tryCount, SetupManger.Open.Move.navigationShowFailSceneTryCountMax, reason,
                countSec,
                { onRetry() },
                { onChangeDest() },
                { onAbort() },
            )
        )
    }

    private fun onRetry() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.RETRY)
        }
    }
    private fun onChangeDest() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.CHANGE_DEST)
        }
    }

    private fun onAbort() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.ABORT)
        }
    }
}


@Composable
fun GuideMoveNavigationShowFailSceneScreen(sceneView: SceneView.GuideMoveNavigationShowFail) {
    MiniImageBg(R.drawable.scene_charge_show_fail_bg) {
        SystemMainText(text = buildAnnotatedString {

            withStyle(SpanStyle(SystemCompose.highlightTextColor)) {
                append(sceneView.destPoi)
            }
            append("(으)로 이동하는데 실패하였습니다.\n")

            withStyle(
                SpanStyle(fontSize = lpToSp(40))
            ) {
                append("(${sceneView.tryCount + 1}/${sceneView.tryCountMax})\n")
            }
            withStyle(
                SpanStyle(Color.Red, lpToSp(50))
            ) {
                val msg = when (val reason = sceneView.reason) {
                    is MoveNavigationStartSceneTask.Result.AlreadyInLocation -> "이미 위치함"
                    is MoveNavigationStartSceneTask.Result.NotExistPoi -> "위치가 존재하지 않음"
                    is MoveNavigationStartSceneTask.Result.UserAbort -> "사용자 취소"
                    is MoveNavigationStartSceneTask.Result.Timeout -> "시간 초과"
                    is MoveNavigationStartSceneTask.Result.StartNavigationResultResult -> {
                        when (reason.startNavigationResult) {
                            is RobotManager.StartNavigationResult.Result -> "Result: (${reason.startNavigationResult.result})"
                            is RobotManager.StartNavigationResult.OnResult -> "OnResult: (${reason.startNavigationResult.status}, ${reason.startNavigationResult.responseString})"
                            is RobotManager.StartNavigationResult.OnError -> "OnError: (${reason.startNavigationResult.errorCode}, ${reason.startNavigationResult.errorString})"
                        }
                    }

                }
                append(msg)
            }
        })

        MiniTextImageButton(162, 900, 516, 132, R.drawable.scene_common_ok_btn,
            "계속 이동하기(${sceneView.countSec})", SystemCompose.textColor, SystemCompose.largeButtonFontSize,
            { sceneView.onRetry() }
        )
        MiniTextImageButton(702, 900, 516, 132, R.drawable.scene_common_cancel_btn,
            "다른 목적지 선택", SystemCompose.textColor, SystemCompose.largeButtonFontSize,
            { sceneView.onChangeDest() }
        )
        MiniTextImageButton(1242, 900, 516, 132, R.drawable.scene_common_cancel_btn,
            "이동 취소", SystemCompose.textColor, SystemCompose.largeButtonFontSize,
            { sceneView.onAbort() }
        )
    }
}
