package com.clobot.minibasic.data.task.mini.loop.mode

import androidx.compose.runtime.Composable
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import com.clobot.minibasic.R
import com.clobot.minibasic.data.RobotManager
import com.clobot.minibasic.data.SetupManger
import com.clobot.minibasic.data.mode.ModeManager
import com.clobot.minibasic.data.operation.OperationManager
import com.clobot.minibasic.data.task.MiniImageBg
import com.clobot.minibasic.data.task.SceneTask
import com.clobot.minibasic.data.task.mini.SystemButton1
import com.clobot.minibasic.data.task.mini.SystemButton2
import com.clobot.minibasic.data.task.mini.SystemCompose
import com.clobot.minibasic.data.task.mini.SystemMainText
import com.clobot.minibasic.data.task.mini.loop.mode.rest.RestChargeBatteryLevel
import com.clobot.minibasic.data.task.mini.loop.mode.rest.RestOperationStateView
import com.clobot.minibasic.view.layer1_scene.SceneView
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.launch

class ShowChangeModeSceneTask(val mode: ModeManager.Mode): SceneTask() {

    enum class Result {
        OK, SKIP
    }
    private val resultChannel = Channel<Result>()

    override val timeoutSec: Int = SetupManger.Open.showChangeModeSceneTimeoutSec

    override suspend fun onRun() : Result {
        return coroutineScope {

            val countJob = launch {
                runCountJob() {
                    updateView()
                }
                resultChannel.send(Result.OK)
            }
            val batteryLevelJob = launch {
                RobotManager.batteryLevelSf.collect {
                    updateView()
                }
            }
            val chargingJob = launch {
                RobotManager.isChargingSf.collect {
                    updateView()
                }
            }
            val operationJob = launch {
                OperationManager.stateSf.collect {
                    updateView()
                    if (it?.mode != mode)
                        resultChannel.send(Result.SKIP)
                }
            }

            updateView()

            try {
                resultChannel.receive()
            } finally {
                operationJob.cancel()
                chargingJob.cancel()
                batteryLevelJob.cancel()
                countJob.cancel()
            }
        }
    }
    private fun updateView() {
        setSceneView(
            SceneView.ShowChangeMode(
                mode,
                RobotManager.batteryLevelSf.value ?: 0,
                RobotManager.isChargingSf.value == true,
                countSec,
                { onStartRest() },
                { onSkipRest() },
                OperationManager.stateSf.value!!,
            )
        )
    }

    private fun onStartRest() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.OK)
        }
    }

    private fun onSkipRest() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.SKIP)
        }
    }
}


@Composable
fun ShowChangeModeSceneScreen(sceneView: SceneView.ShowChangeMode) {
    MiniImageBg(R.drawable.scene_rest) {

        SystemMainText(buildAnnotatedString {
            append("운영 모드가 ")
            withStyle(
                SpanStyle(SystemCompose.highlightTextColor)
            ) {
                append(ModeManager.getModeName(sceneView.mode))
            }
            append("로 바뀔 예정입니다.")
        })


        RestChargeBatteryLevel(sceneView.batteryLevel, sceneView.isCharging)

        SystemButton1(
            "모드 변경(${sceneView.countSec})", sceneView.onStartRest
        )
        SystemButton2(
            "지금은 건너 뜀", sceneView.onSkipRest
        )

        RestOperationStateView(sceneView.operationState)
    }
}


