package com.clobot.minibasic.data.task.mini.loop.charge

import android.widget.VideoView
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.viewinterop.AndroidView
import com.ainirobot.coreservice.client.Definition
import com.clobot.minibasic.R
import com.clobot.minibasic.data.RobotManager
import com.clobot.minibasic.data.SetupManger
import com.clobot.minibasic.data.operation.OperationManager
import com.clobot.minibasic.data.task.MiniCompose
import com.clobot.minibasic.data.task.SceneTask
import com.clobot.minibasic.data.task.lp
import com.clobot.minibasic.data.task.lpToSp
import com.clobot.minibasic.data.task.mini.SystemMainText
import com.clobot.minibasic.data.task.mini.SystemProgressBar
import com.clobot.minibasic.data.task.mini.SystemProgressText
import com.clobot.minibasic.data.task.mini.SystemSubText
import com.clobot.minibasic.data.task.mini.loop.mode.rest.RestOperationStateView
import com.clobot.minibasic.view.layer1_scene.SceneView
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.async
import kotlinx.coroutines.cancelAndJoin
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.launch
import kotlin.coroutines.cancellation.CancellationException

class ChargeStartSceneTask(private val tryCount: Int): SceneTask() {

    private lateinit var startChargeJob: Deferred<RobotManager.StartChargeResult>
    sealed class Result {
        data object Ok : Result()
        class StartChargeResult(val startChargeResult: RobotManager.StartChargeResult) : Result()

        data object UserAbort : Result()
    }


    override val timeoutSec: Int = SetupManger.Rest.Charge.chargeStartSceneTimeoutSec
    override suspend fun onRun(): Result {
        if (isPause)
            return Result.UserAbort
        if (RobotManager.isChargingSf.value!!)
            return Result.Ok

        return coroutineScope {
            val countJob = launch {
                runCountJob () {
                    updateView()
                }
                startChargeJob.cancel()
            }
            val operationJob = launch {
                OperationManager.stateSf.collect() {
                    updateView()
                }
            }

            startChargeJob = async {
                RobotManager.startCharge(SetupManger.Rest.Charge.startChargeTimeoutSec * Definition.SECOND)
            }

            try {
                Result.StartChargeResult(startChargeJob.await())
            } catch (e: CancellationException) {
                Result.UserAbort
            } finally {
                operationJob.cancel()
                countJob.cancel()
            }
        }
    }

    private fun updateView() {
        setSceneView(
            SceneView.ChargeStart(
                tryCount,
                SetupManger.Rest.Charge.chargeShowFailSceneTryCountMax,
                countSec,
                { onPause() },
                OperationManager.stateSf.value!!
            )
        )
    }

    private fun onPause() {
        RobotManager.stopCharge()
        taskCoroutineScope?.launch {
            startChargeJob.cancelAndJoin()
        }
    }
    override fun onPause(isPause: Boolean) {
        if (isPause)
            onPause()
    }
}

@Composable
fun ChargeStartSceneScreen(sceneView: SceneView.ChargeStart) {
    Box(Modifier.lp(0, 0, MiniCompose.bg_width, MiniCompose.bg_height)) {
        val videoPath =
            "android.resource://${LocalContext.current.packageName}/${R.raw.move_mp4}"
        AndroidView(
            modifier = Modifier.fillMaxSize(),
            factory = { mContext ->
                VideoView(mContext).apply {
                    setVideoPath(videoPath)
                    setOnPreparedListener { mediaPlayer ->
                        mediaPlayer.isLooping = true
                    }
                    start()
                }
            }
        )
        SystemMainText(text = buildAnnotatedString {
            append("충전기로 이동 중입니다.")
            withStyle(
                SpanStyle(fontSize = lpToSp(40))
            ) {
                append("(${sceneView.tryCount + 1}/${sceneView.tryCountMax})\n")
            }
            append("지나갈 수 있도록 도와주세요.")
        })

        SystemProgressBar {
            CircularProgressIndicator(modifier = Modifier.fillMaxSize(), color = Color.White)
        }
        SystemProgressText(sceneView.countSec)

        SystemSubText("화면을 터치하면 로봇을 멈출 수 있습니다.")

        RestOperationStateView(sceneView.operationState)

    }
    Box(
        modifier = Modifier
            .fillMaxSize()
            .clickable { sceneView.onPause() }
    ) {
    }
}
