package com.clobot.minibasic.data.task.mini.loop.charge

import androidx.compose.runtime.Composable
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import com.clobot.minibasic.R
import com.clobot.minibasic.data.RobotManager
import com.clobot.minibasic.data.SetupManger
import com.clobot.minibasic.data.operation.OperationManager
import com.clobot.minibasic.data.task.MiniImageBg
import com.clobot.minibasic.data.task.SceneTask
import com.clobot.minibasic.data.task.lpToSp
import com.clobot.minibasic.data.task.mini.SystemButton1
import com.clobot.minibasic.data.task.mini.SystemButton2
import com.clobot.minibasic.data.task.mini.SystemButton3
import com.clobot.minibasic.data.task.mini.SystemMainText
import com.clobot.minibasic.data.task.mini.loop.mode.rest.RestOperationStateView
import com.clobot.minibasic.view.layer1_scene.SceneView
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.launch

class ChargeShowFailSceneTask(private val tryCount: Int, private val reason: ChargeStartSceneTask.Result): SceneTask() {

    enum class Result {
        RETRY, STAY, ABORT,
    }
    private val resultChannel = Channel<Result>()

    override val timeoutSec: Int = SetupManger.Rest.Charge.chargeShowFailSceneTimeoutSec
    override suspend fun onRun(): Result {
        return coroutineScope {
            val countJob = launch {
                runCountJob() {
                    updateView()
                }
                resultChannel.send(if (tryCount < SetupManger.Rest.Charge.chargeShowFailSceneTryCountMax - 1) Result.RETRY else Result.ABORT)
            }
            val operationJob = launch {
                OperationManager.stateSf.collect() {
                    updateView()
                }
            }

            try {
                resultChannel.receive()
            } finally {
                operationJob.cancel()
                countJob.cancel()
            }
        }
    }
    private fun updateView() {
        setSceneView(
            SceneView.ChargeShowFail(tryCount, SetupManger.Rest.Charge.chargeShowFailSceneTryCountMax, reason,
                countSec,
                { onRetry() },
                { onStay() },
                { onAbort() },
                OperationManager.stateSf.value!!,
            )
        )
    }
    private fun onRetry() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.RETRY)
        }
    }

    private fun onStay() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.STAY)
        }
    }
    private fun onAbort() {
        taskCoroutineScope?.launch {
            resultChannel.send(Result.ABORT)
        }
    }
}

@Composable
fun ChargeShowFailSceneScreen(sceneView: SceneView.ChargeShowFail) {
    MiniImageBg(R.drawable.scene_charge_show_fail_bg) {
        SystemMainText(text = buildAnnotatedString {
            append("충전에 실패 하였습니다.")

            withStyle(
                SpanStyle(fontSize = lpToSp(40))
            ) {
                append("(${sceneView.tryCount + 1}/${sceneView.tryCountMax})\n")
            }
            withStyle(
                SpanStyle(Color.Red, lpToSp(50))
            ) {
                val msg = when (val reason = sceneView.reason) {
                    is ChargeStartSceneTask.Result.Ok -> ""
                    is ChargeStartSceneTask.Result.UserAbort -> "사용자 취소"
                    is ChargeStartSceneTask.Result.StartChargeResult -> {
                        when (reason.startChargeResult) {
                            is RobotManager.StartChargeResult.Result -> "Result: (${reason.startChargeResult.result})"
                            is RobotManager.StartChargeResult.OnResult -> "OnResult: (${reason.startChargeResult.status}, ${reason.startChargeResult.responseString})"
                            is RobotManager.StartChargeResult.OnError -> "OnError: (${reason.startChargeResult.errorCode}, ${reason.startChargeResult.errorString})"
                        }
                    }

                }
                append(msg)
            }
        })

        SystemButton1(
            "재시도" + if (sceneView.tryCount < sceneView.tryCountMax - 1) "(${sceneView.countSec})" else "",
            sceneView.onRetry
        )
        SystemButton2(
            "대기" + if (sceneView.tryCount >= sceneView.tryCountMax - 1) "(${sceneView.countSec})" else "",
            sceneView.onStay,
        )
        SystemButton3(
            "취소",
            sceneView.onAbort,
        )

        RestOperationStateView(sceneView.operationState)
    }
}
