package com.clobot.minibasic.data.task

import android.annotation.SuppressLint
import androidx.annotation.DrawableRes
import androidx.compose.animation.animateColor
import androidx.compose.animation.core.RepeatMode
import androidx.compose.animation.core.infiniteRepeatable
import androidx.compose.animation.core.rememberInfiniteTransition
import androidx.compose.animation.core.tween
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.foundation.text.InlineTextContent
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.ReadOnlyComposable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.composed
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.graphics.BlendMode
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.graphics.DefaultAlpha
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.PlatformTextStyle
import androidx.compose.ui.text.TextLayoutResult
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.Font
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.clobot.minibasic.R
import com.clobot.minibasic.data.ProfileManager
import com.clobot.minibasic.data.task.MiniCompose.Companion.debugBorder


class MiniCompose {
    companion object {
        const val bg_width = 1920
        const val bg_height = 1080
        const val bg_center_x = bg_width / 2
        const val bg_center_y = bg_height / 2

        private const val CLASS_NAME = "MiniCompose::"
        private const val IS_DEBUG_BORDER_KEY = CLASS_NAME + "isDebugBorder"
        var isDebugBorder = ProfileManager.getPutBoolean(IS_DEBUG_BORDER_KEY, false)
            set(value) {
                ProfileManager.putBoolean(IS_DEBUG_BORDER_KEY, value)
                field = value
            }

        val debugBorder: BorderStroke = BorderStroke(if (isDebugBorder) 0.dp else (-1).dp, Color(0xFF6650a4))

        val minSansFont = FontFamily(
            Font(R.font.minsans_thin, FontWeight.Thin, FontStyle.Normal),
            Font(R.font.minsans_extralight, FontWeight.ExtraLight, FontStyle.Normal),
            Font(R.font.minsans_light, FontWeight.Light, FontStyle.Normal),
            Font(R.font.minsans_regular, FontWeight.Normal, FontStyle.Normal),
            Font(R.font.minsans_medium, FontWeight.Medium, FontStyle.Normal),
            Font(R.font.minsans_semibold, FontWeight.SemiBold, FontStyle.Normal),
            Font(R.font.minsans_bold, FontWeight.Bold, FontStyle.Normal),
            Font(R.font.minsans_extrabold, FontWeight.ExtraBold, FontStyle.Normal),
            Font(R.font.minsans_black, FontWeight.Black, FontStyle.Normal),
        )
    }
}


@Composable
@ReadOnlyComposable
fun lpXToDpX(lpx: Int): Dp {
    val context = LocalContext.current
    return Dp(lpx / MiniCompose.bg_width.toFloat() * context.resources.displayMetrics.widthPixels / context.resources.displayMetrics.density)
}

@Composable
@ReadOnlyComposable
fun lpYToDpY(lpy: Int): Dp {
    val context = LocalContext.current
    return Dp(lpy / MiniCompose.bg_height.toFloat() * context.resources.displayMetrics.heightPixels / context.resources.displayMetrics.density)
}

@Composable
@ReadOnlyComposable
fun lpWToDpW(lpw: Int): Dp = lpXToDpX(lpw)

@Composable
@ReadOnlyComposable
fun lpHToDpH(lph: Int): Dp = lpYToDpY(lph)


@Composable
@ReadOnlyComposable
fun lpToSp(lp: Int): TextUnit {
    val context = LocalContext.current
    return (lpYToDpY(lp).value * context.resources.displayMetrics.density / context.resources.displayMetrics.scaledDensity).sp
}

@SuppressLint("ModifierFactoryUnreferencedReceiver")
fun Modifier.lp(x: Int, y: Int, width: Int, height: Int): Modifier = composed {
    offset(lpXToDpX(x), lpYToDpY(y))
        .size(lpWToDpW(width), lpHToDpH(height))
        .border(debugBorder)
}
@SuppressLint("ModifierFactoryUnreferencedReceiver")
fun Modifier.lpOffset(width: Int, height: Int): Modifier = composed {
    offset(lpWToDpW(width), lpHToDpH(height))
        .border(debugBorder)
}

@SuppressLint("ModifierFactoryUnreferencedReceiver")
fun Modifier.lpSize(width: Int, height: Int): Modifier = composed {
    size(lpWToDpW(width), lpHToDpH(height))
    .border(debugBorder)
}

@Composable
fun MiniImageBg(@DrawableRes bgId: Int, contents: @Composable () -> Unit) {
    Box(Modifier.lp(0, 0, MiniCompose.bg_width, MiniCompose.bg_height)) {
        Image(
            modifier = Modifier.fillMaxSize(),
            painter = painterResource(bgId),
            contentDescription = "",
            contentScale = ContentScale.FillBounds
        )
        contents()
    }
}


@Composable
fun MiniText(
    text: AnnotatedString,
    modifier: Modifier,
    color: Color,
    fontSize: TextUnit,
    fontStyle: FontStyle? = FontStyle.Normal,
    fontWeight: FontWeight? = FontWeight.Bold,
    fontFamily: FontFamily = MiniCompose.minSansFont,
    letterSpacing: TextUnit = TextUnit.Unspecified,
    textDecoration: TextDecoration? = null,
    textAlign: TextAlign? = TextAlign.Center,
    lineHeight: TextUnit = TextUnit.Unspecified,
    overflow: TextOverflow = TextOverflow.Visible,
    softWrap: Boolean = true,
    maxLines: Int = Int.MAX_VALUE,
    minLines: Int = 1,
    inlineContent: Map<String, InlineTextContent> = mapOf(),
    onTextLayout: (TextLayoutResult) -> Unit = {},
    style: TextStyle = TextStyle(
        platformStyle = PlatformTextStyle(
            includeFontPadding = false
        ),
    )
) {
    Text(
        text,
        modifier,
        color,
        fontSize,
        fontStyle,
        fontWeight,
        fontFamily,
        letterSpacing,
        textDecoration,
        textAlign,
        lineHeight,
        overflow,
        softWrap,
        maxLines,
        minLines,
        inlineContent,
        onTextLayout,
        style)
}
@Composable
fun MiniText(
    text: String,
    modifier: Modifier = Modifier,
    color: Color = Color.Unspecified,
    fontSize: TextUnit = TextUnit.Unspecified,
    fontStyle: FontStyle? = FontStyle.Normal,
    fontWeight: FontWeight? = FontWeight.Bold,
    fontFamily: FontFamily? = MiniCompose.minSansFont,
    letterSpacing: TextUnit = TextUnit.Unspecified,
    textDecoration: TextDecoration? = null,
    textAlign: TextAlign? = TextAlign.Center,
    lineHeight: TextUnit = TextUnit.Unspecified,
    overflow: TextOverflow = TextOverflow.Visible,
    softWrap: Boolean = true,
    maxLines: Int = Int.MAX_VALUE,
    minLines: Int = 1,
    onTextLayout: (TextLayoutResult) -> Unit = {},
    style: TextStyle = TextStyle(
        platformStyle = PlatformTextStyle(
            includeFontPadding = false
        ),
    )
) {
    Text(
        text,
        modifier,
        color,
        fontSize,
        fontStyle,
        fontWeight,
        fontFamily,
        letterSpacing,
        textDecoration,
        textAlign,
        lineHeight,
        overflow,
        softWrap,
        maxLines,
        minLines,
        onTextLayout,
        style)
}

@Composable
fun MiniImage(
    resId: Int,
    modifier: Modifier,
    alignment: Alignment = Alignment.Center,
    contentScale: ContentScale = ContentScale.FillBounds,
    alpha: Float = DefaultAlpha,
    colorFilter: ColorFilter? = null
) {
    Image(
        painterResource(id = resId),
        "",
        modifier,
        alignment,
        contentScale,
        alpha,
        colorFilter
    )
}

@Composable
fun MiniFitText(x: Int, y: Int, w: Int, h: Int, text: AnnotatedString, color: Color, fontSize: Int = 0, textAlign: TextAlign? = TextAlign.Center, extraW: Int = 20, extraH: Int = 20, alignment: Alignment = Alignment.Center, alpha: Float = 1.0f) {
    Box() {
        Box(Modifier.lp(x, y, w, h)) {
        }
        Text(
            modifier = Modifier
                .offset(lpXToDpX(x - extraW), lpYToDpY(y - extraH))
                .size(lpWToDpW(w + extraW * 2), lpHToDpH(h + extraH * 2))
                .border(debugBorder).alpha(alpha)
                .wrapContentSize(alignment),
            color = color,
            fontSize = lpToSp(if (fontSize == 0) h else fontSize),
            text = text,
            textAlign = textAlign,
            fontFamily = MiniCompose.minSansFont,
            fontWeight = FontWeight.Bold,
            overflow = TextOverflow.Visible,
            style = TextStyle(
                platformStyle = PlatformTextStyle(
                    includeFontPadding = false
                ),
            )
        )
    }
}
@Composable
fun MiniFitText(x: Int, y: Int, w: Int, h: Int, text: String, color: Color, fontSize: Int = 0, textAlign: TextAlign? = TextAlign.Center, extraW: Int = 20, extraH: Int = 20, alignment: Alignment = Alignment.Center) {
    MiniFitText(x, y, w, h, AnnotatedString(text), color, fontSize, textAlign, extraW, extraH)
}

@Composable
fun MiniTextImageButton(x: Int, y: Int, w: Int, h: Int, @DrawableRes resId: Int, text: String, color: Color, fontSize: Int, onClick: () -> Unit, isAnimation: Boolean = false) {
    Box(
        Modifier
            .lp(x, y, w, h)
            .clickable { onClick() }) {
        if (isAnimation) {
            val infiniteTransition = rememberInfiniteTransition(label = "")
            val animatedColor by infiniteTransition.animateColor(
                initialValue = Color(0xffff3030),
                targetValue = Color(0x000000),
                animationSpec = infiniteRepeatable(tween(1000), RepeatMode.Reverse),
                label = ""
            )
            MiniImage(
                resId,
                Modifier.fillMaxSize(),
                colorFilter = ColorFilter.tint(animatedColor, blendMode = BlendMode.Darken)
            )
        }
        else
            MiniImage(
                resId,
                Modifier.fillMaxSize(),
            )
        MiniText(
            text,
            Modifier
                .fillMaxSize()
                .wrapContentSize(),
            color,
            lpToSp(fontSize),
        )
    }
}




/*

@Composable
fun LpBoxButton(
    x: Int,
    y: Int,
    width: Int,
    height: Int,
    contentAlignment: Alignment = Alignment.TopStart,
    onClick: (() -> Unit)?,
    alpha: Float = 1.0f,
    contents: @Composable () -> Unit = {}
) {

    Box(
        modifier = if (onClick != null) {
            Modifier
                .offset(x = lpXToDpX(x), y = lpYToDpY(y))
                .size(lpWToDpW(width), lpHToDpH(height))
                .border(PrcCompose.debugBorder)
                .clickable(enabled = true, onClick = { onClick!!() })
                .alpha(alpha)
        } else {
            Modifier
                .offset(x = lpXToDpX(x), y = lpYToDpY(y))
                .size(lpWToDpW(width), lpHToDpH(height))
                .border(PrcCompose.debugBorder)
                .alpha(alpha)
        },
        contentAlignment = contentAlignment
    ) {
        contents()
    }
}

@Composable
fun LpRoundBoxButton(
    x: Int,
    y: Int,
    width: Int,
    height: Int,
    borderColor: Color,
    borderWidth: Dp,
    borderCornerSize: Dp,
    onClick: (() -> Unit)?,
    contents: @Composable () -> Unit
) {
    Box(
        modifier = Modifier
            .offset(x = lpXToDpX(x), y = lpYToDpY(y))
            .size(lpWToDpW(width), lpHToDpH(height))
            .background(Color(0f, 0f, 0f, alpha = 0.5f))
            .border(BorderStroke(borderWidth, borderColor), RoundedCornerShape(borderCornerSize))
            .clickable(enabled = onClick != null) {
                onClick!!()
            },
        contentAlignment = Alignment.Center,
    ) {
        contents()
    }
}

@Composable
fun LpBox(
    x: Int,
    y: Int,
    width: Int,
    height: Int,
    contentAlignment: Alignment = Alignment.TopStart,
    contents: @Composable () -> Unit = {}
) {
    LpBoxButton(x, y, width, height, contentAlignment, null, 1.0f, contents)
}

@Composable
fun LpImageButton(
    x: Int,
    y: Int,
    width: Int,
    height: Int,
    @DrawableRes id: Int,
    onClick: (() -> Unit)?,
    alpha: Float = DefaultAlpha,
    colorFilter: ColorFilter? = null
) {
    Image(
        painter = painterResource(id),
        contentDescription = "",
        contentScale = ContentScale.FillBounds,
        modifier = if (onClick != null) {
            Modifier
                .offset(x = lpXToDpX(x), y = lpYToDpY(y))
                .size(lpWToDpW(width), lpHToDpH(height))
                .border(PrcCompose.debugBorder)
                .clickable(enabled = true, onClick = { onClick!!() })
        } else {
            Modifier
                .offset(x = lpXToDpX(x), y = lpYToDpY(y))
                .size(lpWToDpW(width), lpHToDpH(height))
                .border(PrcCompose.debugBorder)
        },
        alpha = alpha,
        colorFilter = colorFilter,

        )
}


@Composable
fun LpImage(x: Int, y: Int, width: Int, height: Int, @DrawableRes id: Int) {
    LpImageButton(x, y, width, height, id, onClick = null)
}
*/


@Composable
fun MiniBlackBg(contents: @Composable () -> Unit) {
    Box(
        Modifier
            .fillMaxSize()
            .background(Color.Black)
    ) {
        contents()
    }
}


/*
@Composable
fun LpText(
    x: Int,
    y: Int,
    w: Int,
    h: Int,
    text: AnnotatedString,
    color: Color,
    fontSize: Int,
    textAlign: TextAlign? = TextAlign.Center,
    fontWeight: FontWeight? = FontWeight.Bold,
    alignment: Alignment = Alignment.Center
) {
    LpBox(x, y, w, h) {
        Text(
            modifier = Modifier
                .fillMaxSize()
                .wrapContentSize(alignment),
            color = color,
            fontSize = lpToSp(lp = fontSize),
            text = text,
            textAlign = textAlign,
            fontFamily = PrcCompose.minSansFont,
            fontWeight = fontWeight,
            overflow = TextOverflow.Visible,
            style = TextStyle(
                platformStyle = PlatformTextStyle(
                    includeFontPadding = false
                ),
            )
        )
    }
}

@Composable
fun LpText(x: Int, y: Int, w: Int, h: Int, text: String, color: Color, fontSize: Int) {
    LpText(x, y, w, h, AnnotatedString(text), color, fontSize)
}


@Composable
fun LpText(x: Int, y: Int, w: Int, h: Int, textId: Int, color: Color, fontSize: Int) {
    LpText(x, y, w, h, AnnotatedString(stringResource(textId)), color, fontSize)
}
*/

/*
@Composable
fun LpImageTextButton(
    x: Int,
    y: Int,
    w: Int,
    h: Int,
    @DrawableRes id: Int,
    text: String,
    color: Color,
    fontSize: Int,
    onClick: (() -> Unit)?,
) {
    LpBoxButton(x, y, w, h, onClick = onClick, alpha = if (onClick != null) 1.0f else 0.3f) {
        Image(
            modifier = Modifier.fillMaxSize(),
            painter = painterResource(id),
            contentDescription = "",
            contentScale = ContentScale.FillBounds,
        )
        Text(
            modifier = Modifier
                .fillMaxSize()
                .wrapContentSize(),
            text = text,
            color = color,
            fontSize = lpToSp(fontSize),
            fontFamily = PrcCompose.minSansFont,
            fontWeight = FontWeight.Bold,
            textAlign = TextAlign.Center,
            overflow = TextOverflow.Visible,
        )
    }
}
*/