package com.clobot.minibasic.data.operation

import androidx.compose.runtime.Immutable
import com.clobot.minibasic.data.ClockManager
import com.clobot.minibasic.data.ProfileManager
import com.clobot.minibasic.data.mode.ModeManager
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import java.time.DayOfWeek
import java.time.LocalDateTime


class ScheduleChecker {

    companion object {
        private const val CLASS_NAME = "ScheduleChecker::"

        private const val MODE_KEY = CLASS_NAME + "Mode"

        private const val DAY_ITEM_IS_ON_KEY = CLASS_NAME + "dayItemIsOn"
        private const val DAY_ITEM_SLOT_KEY = CLASS_NAME + "dayItemSlot"

        private const val MODE_TIME_COUNT_KEY = CLASS_NAME + "modeTimeCount"
        private const val MODE_TIME_ITEM_MODE_KEY = CLASS_NAME + "modeTimeItemModeKey"
        private const val MODE_TIME_ITEM_RANGE_FIRST_KEY = CLASS_NAME + "modeTimeItemRangeFirstKey"
        private const val MODE_TIME_ITEM_RANGE_LAST_KEY = CLASS_NAME + "modeTimeItemRangeLastKey"
    }

    enum class Mode {
        ALWAYS_REST,
        ALWAYS_GUIDE,
        ALWAYS_PROMOTE,
        SCHEDULE,
   }

    var mode = Mode.entries[ProfileManager.getPutInt(MODE_KEY, 0)]
        set(value) {
            ProfileManager.putInt(MODE_KEY, value.ordinal)
            field = value
        }

    private var dateTime = LocalDateTime.now()
        set(value) {
            field = value
            update()
        }

    sealed class State(val mode: ModeManager.Mode) {
        data object AlwaysRest : State(ModeManager.Mode.REST)
        data object AlwaysGuide : State(ModeManager.Mode.GUIDE)
        data object AlwaysPromote : State(ModeManager.Mode.PROMOTE)
        data object ScheduleDayRest : State(ModeManager.Mode.REST)
        class Schedule(mode: ModeManager.Mode, val lastTime: Int) : State(mode)
    }

    private fun check(): State {
        return when (mode) {
            Mode.ALWAYS_REST -> State.AlwaysRest
            Mode.ALWAYS_GUIDE -> State.AlwaysGuide
            Mode.ALWAYS_PROMOTE -> State.AlwaysPromote
            Mode.SCHEDULE -> {
                if (dayDataArr[dateTime.dayOfWeek.ordinal].isOn) {
                    val t = dateTime.hour * 60 * 60 + dateTime.minute * 60 + dateTime.second
                    val modeTimeDataArr = modeTimeDataListArr[dayDataArr[dateTime.dayOfWeek.ordinal].slot]
                    modeTimeDataArr.forEach { modeTimeData ->
                        if (t >= modeTimeData.timeRange.first && t < modeTimeData.timeRange.last)
                            return State.Schedule(modeTimeData.mode, modeTimeData.timeRange.last)
                    }

                    var minLastTime = 24 * 60 * 60
                    modeTimeDataArr.forEach { modeTimeData ->
                        if (t < modeTimeData.timeRange.first && modeTimeData.timeRange.first < minLastTime)
                            minLastTime = modeTimeData.timeRange.first
                    }
                    State.Schedule(ModeManager.Mode.REST, minLastTime)
                } else
                    State.ScheduleDayRest
            }
        }
    }

    data class DayData(val isOn: Boolean, val slot: Int)
    val dayDataArr: Array<DayData> = Array(DayOfWeek.entries.size) { day ->
        val isOn = ProfileManager.getPutBoolean(DAY_ITEM_IS_ON_KEY + day, false)
        val slot = ProfileManager.getPutInt(DAY_ITEM_SLOT_KEY + day, 0)
        DayData(isOn, slot)
    }
    fun saveDayArr() {
        dayDataArr.forEachIndexed { day, dayData ->
            ProfileManager.putBoolean(
                DAY_ITEM_IS_ON_KEY + day,
                dayData.isOn
            )
            ProfileManager.putInt(
                DAY_ITEM_SLOT_KEY + day,
                dayData.slot
            )
        }
    }


    data class ModeTimeData(val mode: ModeManager.Mode, val timeRange: IntRange)
    val modeTimeDataListArr: Array<MutableList<ModeTimeData>> = Array(5) { slot ->
        val count = ProfileManager.getPutInt(MODE_TIME_COUNT_KEY + slot, 0)
        MutableList(count) { index ->
            val mode = ModeManager.Mode.entries[ProfileManager.getPutInt(MODE_TIME_ITEM_MODE_KEY + slot + "_$index", 0)]
            val first = ProfileManager.getPutInt(MODE_TIME_ITEM_RANGE_FIRST_KEY + slot + "_$index", 0)
            val last = ProfileManager.getPutInt(MODE_TIME_ITEM_RANGE_LAST_KEY + slot + "_$index", 1)
            ModeTimeData(mode, first .. last)
        }
    }
    fun saveModeTimeDataListArr() {
        modeTimeDataListArr.forEachIndexed { slot, modeTimeDataList ->
            ProfileManager.putInt(
                MODE_TIME_COUNT_KEY + slot,
                modeTimeDataList.size
            )
            modeTimeDataList.forEachIndexed { index, scheduleModeTimeData ->
                ProfileManager.putInt(
                    MODE_TIME_ITEM_MODE_KEY + slot + "_$index",
                    scheduleModeTimeData.mode.ordinal
                )
                ProfileManager.putInt(
                    MODE_TIME_ITEM_RANGE_FIRST_KEY + slot + "_$index",
                    scheduleModeTimeData.timeRange.first
                )
                ProfileManager.putInt(
                    MODE_TIME_ITEM_RANGE_LAST_KEY + slot + "_$index",
                    scheduleModeTimeData.timeRange.last
                )
            }
        }
    }


    /*
    private fun testModeTimeData() {
        repeat(DayOfWeek.entries.size) { i ->
            scheduleDayDataArr[i].isOn = true
            scheduleDayDataArr[i].scheduleSlot = 0
        }
        repeat(scheduleModeTimeDataListArr.size) { i ->
            scheduleModeTimeDataListArr[i].clear()
            scheduleModeTimeDataListArr[i].add(
                ScheduleModeTimeData(
                    ModeManager.Mode.GUIDE, 9 * 3600  .. 18 * 3600
                )
            )
        }
    }
    private fun initModeTimeData() {
        repeat(DayOfWeek.entries.size) { i ->
            scheduleModeTimeDayDataArr[i].isOn = true
            scheduleModeTimeDayDataArr[i].scheduleModeTimeDataList.clear()
            val dayCount = ProfileManager.getPutInt(SCHEDULE_MODE_TIME_DAY_COUNT_KEY + i.toString(), 0)
            repeat(dayCount) { j ->
                val key = i.toString() + "_" + j.toString()
                scheduleModeTimeDayDataArr[i].scheduleModeTimeDataList.add(
                    ScheduleModeTimeData(
                        ModeManager.Mode.entries[ProfileManager.getPutInt(
                            SCHEDULE_MODE_TIME_ITEM_MODE_KEY + key, 0)],
                        ProfileManager.getPutInt(SCHEDULE_MODE_TIME_ITEM_RANGE_FIRST_KEY + key, 0) ..
                                ProfileManager.getPutInt(SCHEDULE_MODE_TIME_ITEM_RANGE_LAST_KEY + key, 0)
                    )
                )
            }
        }
    }

     */

    private val stateOpenMsf = MutableStateFlow(check())
    val stateSf = stateOpenMsf.asStateFlow()

    private fun update() {
        stateOpenMsf.update {
            check()
        }
    }

    fun init(localDateTime: LocalDateTime) {
        dateTime = localDateTime

        //testModeTimeData()
        update()
        MainScope().launch {
            ClockManager.dateTimeSf.collect {
                dateTime = it
            }
        }
    }
}