package com.clobot.minibasic.data.admin.tab

import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material3.Slider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.dp
import com.ainirobot.coreservice.client.RobotApi
import com.clobot.minibasic.data.RobotManager
import com.clobot.minibasic.data.SetupManger
import com.clobot.minibasic.data.admin.AdminManager
import com.clobot.minibasic.data.admin.AdminSmallText
import com.clobot.minibasic.data.admin.AdminTab
import com.clobot.minibasic.data.admin.AdminText
import com.clobot.minibasic.view.layer5_admin.AdminTabView
import kotlin.math.roundToInt

class RobotAdminTab(adminManager: AdminManager) : AdminTab(adminManager) {
    override val name: String
        get() = "로봇"

    override val adminTabView: AdminTabView
        get() {
            val poiList = mutableListOf<String>()
            for (place in RobotManager.placeList)
                poiList.add(place.name)

            return AdminTabView.Robot(
                SetupManger.Robot.startNavigationLinearSpeed,
                { onLinearSpeed(it) },
                SetupManger.Robot.startNavigationAngularSpeed,
                { onAngularSpeed(it) },
                SetupManger.Robot.startNavigationTimeoutSec,
                { onTimeoutSec(it) },
                poiList.toTypedArray(),
                arrayOf(SetupManger.Open.Move.initPosPoi, /* DentalModeManager.Guide.callPosPoi*/),
            )
        }

    private fun onLinearSpeed(linearSpeed: Float) {
        SetupManger.Robot.startNavigationLinearSpeed = linearSpeed
        updateAdminTabView()
    }
    private fun onAngularSpeed(angularSpeed: Float) {
        SetupManger.Robot.startNavigationAngularSpeed = angularSpeed
        updateAdminTabView()
    }
    private fun onTimeoutSec(timeoutSec: Int) {
        SetupManger.Robot.startNavigationTimeoutSec = timeoutSec
        updateAdminTabView()
    }
}



@Composable
fun RobotAdminTabScreen(adminTabView: AdminTabView.Robot) {

    var linearSpeed by remember { mutableFloatStateOf(adminTabView.linearSpeed) }
    var angularSpeed by remember { mutableFloatStateOf(adminTabView.angularSpeed) }
    var timeout by remember { mutableIntStateOf(adminTabView.timeout) }

    LazyColumn(contentPadding = PaddingValues(horizontal = 30.dp, vertical = 10.dp)) {
        item {
            AdminText(buildAnnotatedString{
                append("이동 직선 속도: ")
                withStyle(SpanStyle(AdminManager.highlightTextColor)) {
                    append(String.format("%.2f", linearSpeed))
                }
                append(" m/s")
                withStyle(SpanStyle(AdminManager.warningTextColor)) {
                    append("   (경고) 속도가 너무 높으면 위험할 수 있습니다.")
                }
            })
        }
        item {
            val min = 0.1f
            val width = 1.0f
            val widthUnit = 0.1f
            Box() {
                Slider(
                    value = (linearSpeed - min) / width,
                    onValueChange = { linearSpeed = it * width + min },
                    onValueChangeFinished = {
                        adminTabView.onLinearSpeed(
                            linearSpeed
                        )
                    },
                    steps = (width / widthUnit).roundToInt() - 1,
                )
                Row(modifier = Modifier
                    .fillMaxSize()
                    .align(Alignment.BottomStart), horizontalArrangement = Arrangement.SpaceBetween) {
                    AdminSmallText(String.format("%.1f", min))
                    AdminSmallText(String.format("%.1f", min + width))
                }
            }
        }

        item {
            AdminText(buildAnnotatedString{
                append("이동 회전 속도: ")
                withStyle(SpanStyle(AdminManager.highlightTextColor)) {
                    append(String.format("%.2f", angularSpeed))
                }
                append(" rad/s")
            })
        }
        item {
            val min = 0.1f
            val width = 3.0f
            val widthUnit = 0.1f
            Box() {
                Slider(value = (angularSpeed - min) / width,
                    onValueChange = {angularSpeed = it * width + min},
                    onValueChangeFinished = {adminTabView.onAngularSpeed(angularSpeed)},
                    steps = (width / widthUnit).roundToInt() - 1,
                )
                Row(modifier = Modifier
                    .fillMaxSize()
                    .align(Alignment.BottomStart), horizontalArrangement = Arrangement.SpaceBetween) {
                    AdminSmallText(String.format("%.1f", min))
                    AdminSmallText(String.format("%.1f", min + width))
                }
            }
        }

        item {
            AdminText(buildAnnotatedString{
                append("회피 타임 아웃: ")
                withStyle(SpanStyle(AdminManager.highlightTextColor)) {
                    append(timeout.toString())
                }
                append(" sec")
            })
        }


        item {
            val min = 10
            val width = 180
            val widthUnit = 5
            Box() {
                Slider(
                    value = (timeout - min).toFloat() / width,
                    onValueChange = { timeout = (it * width).roundToInt() + min },
                    onValueChangeFinished = { adminTabView.onTimeout(timeout) },
                    steps = width / widthUnit - 1,
                )
                Row(modifier = Modifier
                    .fillMaxSize()
                    .align(Alignment.BottomStart), horizontalArrangement = Arrangement.SpaceBetween) {
                    AdminSmallText(min.toString())
                    AdminSmallText((min + width).toString())
                }
            }
        }

        item {
            AdminText("POI 목록: ", color = Color.White)
        }
        item {
            LazyColumn(modifier = Modifier
                .fillMaxWidth()
                .height(150.dp)
                .border(1.dp, Color.Gray)) {
                for (poi in adminTabView.poiArray) {
                    item {
                        AdminText(poi)
                    }
                }
            }
        }
        item {
            AdminText("필수 POI 목록:", color = Color.White)
        }
        item {
            LazyColumn(modifier = Modifier
                .fillMaxWidth()
                .height(150.dp)
                .border(1.dp, Color.Gray)) {
                for (poi in adminTabView.usedPoiArray) {
                    item {
                        AdminText(poi)
                    }
                }
            }
        }
    }
}
