package com.clobot.minibasic.data

import android.content.Context
import android.os.RemoteException
import com.ainirobot.coreservice.client.ApiListener
import com.ainirobot.coreservice.client.Definition
import com.ainirobot.coreservice.client.RobotApi
import com.ainirobot.coreservice.client.StatusListener
import com.ainirobot.coreservice.client.actionbean.Pose
import com.ainirobot.coreservice.client.listener.ActionListener
import com.ainirobot.coreservice.client.listener.CommandListener
import com.ainirobot.coreservice.client.listener.TextListener
import com.ainirobot.coreservice.client.person.PersonApi
import com.ainirobot.coreservice.client.person.PersonListener
import com.ainirobot.coreservice.client.speech.SkillApi
import com.ainirobot.coreservice.client.speech.SkillCallback
import com.ainirobot.coreservice.client.speech.entity.TTSEntity
import com.ainirobot.coreservice.client.speech.entity.TTSParams.SPEECHVOLUME
import com.ainirobot.coreservice.client.speech.entity.TTSParams.SpeechVolume.*
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.suspendCancellableCoroutine
import org.json.JSONException
import org.json.JSONObject
import kotlin.concurrent.timer
import kotlin.coroutines.resume

object RobotManager {

    private val isEmergencyMsf = MutableStateFlow<Boolean?>(null)
    val isEmergencySf = isEmergencyMsf.asStateFlow()
    private val isChargingMsf = MutableStateFlow<Boolean?>(null)
    val isChargingSf = isChargingMsf.asStateFlow()
    private val batteryLevelMsf = MutableStateFlow<Int?>(null)
    val batteryLevelSf = batteryLevelMsf.asStateFlow()
    private val isPersonMsf = MutableStateFlow<Boolean?>(null)
    val isPersonSf = isPersonMsf.asStateFlow()
    private val logMsf = MutableStateFlow<String?>(null)
    val logSf = logMsf.asStateFlow()

    private var isRobot = true
    private var isPersonNullTest = true

    suspend fun init(context: Context, isRobot: Boolean = true, isPersonNullTest: Boolean = false): Boolean {

        this.isRobot = isRobot
        this.isPersonNullTest = isPersonNullTest

        if (!this.isRobot) {
            isEmergencyMsf.value = false
            isChargingMsf.value = false
            batteryLevelMsf.value = 99
            isPersonMsf.value = true
            delay(1000)
            return true
        }
        else {
            initRobotApi(context)
            var activeCount = 30
            while (activeCount > 0) {
                delay(300)
                val robotApi = RobotApi.getInstance()
                if (robotApi.isApiConnectedService && robotApi.isActive)
                    break
                activeCount--
            }
            return if (activeCount > 0) {

                getChargingAndBatteryLevelStatus()
                getEmergencyStatus()

                var stateCount = 30
                while (stateCount > 0) {
                    delay(300)
                    if (isEmergencyMsf.value != null && isChargingSf.value != null && batteryLevelSf.value != null)
                        break
                    stateCount--
                }
                stateCount > 0
            } else
                false
        }
    }

    private fun initRobotApi(context: Context) {
        val robotApi = RobotApi.getInstance()
        try {
            robotApi.connectServer(context, object : ApiListener {
                override fun handleApiDisabled() {
                }

                override fun handleApiConnected() {
                    robotApi.disableBattery()
                    robotApi.disableEmergency()

                    registerEmergencyCallback()
                    registerChargingAndBatteryLevelCallback()

                    if(!isPersonNullTest)
                        registerPersonListener()

                    registerSkillApiListener(context)

                }
                override fun handleApiDisconnected() {
                }
            })
        }
        catch (e: Exception) {
            e.printStackTrace()
        }
    }


    private fun getEmergencyStatus() {
        RobotApi.getInstance().getEmergencyStatus(0, object : CommandListener() {
            override fun onResult(result: Int, message: String?, extraData: String?) {
                isEmergencyMsf.update {
                    message == "1"
                }
            }
        })
    }

    private fun registerEmergencyCallback() {
        RobotApi.getInstance().registerStatusListener(Definition.STATUS_EMERGENCY, object : StatusListener() {
            override fun onStatusUpdate(type: String?, data: String?) {
                super.onStatusUpdate(type, data)
                isEmergencyMsf.update {
                    data == "1"
                }
            }
        })
    }

    private fun getChargingAndBatteryLevelStatus() {
        RobotApi.getInstance().getRobotStatus(Definition.STATUS_BATTERY, object : StatusListener() {
            override fun onStatusUpdate(type: String?, data: String?) {
                try {
                    val jsonObject = JSONObject(data!!)
                    batteryLevelMsf.update {
                        jsonObject.optInt("level", 0)
                    }
                    isChargingMsf.update {
                        jsonObject.optBoolean("isCharging", false)
                    }
                } catch (e: JSONException) {
                    //Timber.e("type: $type, data: $data")
                }
            }
        })
    }

    private fun registerChargingAndBatteryLevelCallback() {
        RobotApi.getInstance().registerStatusListener(Definition.STATUS_BATTERY, object : StatusListener() {
            override fun onStatusUpdate(type: String?, data: String?) {
                try {
                    val jsonObject = JSONObject(data!!)
                    batteryLevelMsf.update {
                        jsonObject.optInt("level", 0)
                    }
                    isChargingMsf.update {
                        jsonObject.optBoolean("isCharging", false)
                    }
                } catch (e: JSONException) {
                    //Timber.e("type: $type, data: $data")
                }
            }
        })
    }


    var getCompleteFaceListMaxDistance = 0.7
    fun registerPersonListener() {
        PersonApi.getInstance().registerPersonListener(object : PersonListener() {
            override fun personChanged() {
                super.personChanged()

                val personList = PersonApi.getInstance().getCompleteFaceList(getCompleteFaceListMaxDistance)

                isPersonMsf.update {
                    personList != null && personList.size > 0
                }
            }
        })
    }
    fun unregisterPersonListener() {
        PersonApi.getInstance().registerPersonListener(null)
    }



    private val _skillApi = SkillApi()
    private val _skillCallback = object: SkillCallback() {
        override fun onSpeechParResult(p0: String?) {
        }

        override fun onStart() {
        }

        override fun onStop() {
        }

        override fun onVolumeChange(p0: Int) {
        }

        override fun onQueryEnded(p0: Int) {
        }
    }



    fun registerSkillApiListener(context: Context) {
        _skillApi.addApiEventListener(object: ApiListener {
            override fun handleApiDisabled() {
            }

            override fun handleApiConnected() {
                _skillApi.registerCallBack(_skillCallback)
            }

            override fun handleApiDisconnected() {
            }

        })
        _skillApi.connectApi(context)
    }



    enum class PlayTextResult {
        COMPLETE, STOP, ERROR,
    }
    suspend fun playText(text:String, volume:Float) = suspendCancellableCoroutine{
        if (isRobot) {
            val ttsParam = LinkedHashMap<String, String>()
            ttsParam[SPEECHVOLUME] = (volume * 30).toInt().toString()
            _skillApi.playText(TTSEntity(null, text, ttsParam), object : TextListener() {
                override fun onStop() {
                    super.onStop()
                    it.resume(PlayTextResult.STOP)
                }

                override fun onComplete() {
                    super.onComplete()
                    it.resume(PlayTextResult.COMPLETE)
                }

                override fun onError() {
                    super.onError()
                    it.resume(PlayTextResult.ERROR)
                }
            })
        } else {
            timer(period = 1000, initialDelay = 1000) {
                cancel()
                it.resume(PlayTextResult.COMPLETE)
            }
        }

    }
    fun stopText() {
        if (isRobot)
            _skillApi.stopTTS()
    }





    sealed class StartChargeResult {
        class Result(val result: Int) : StartChargeResult()
        class OnResult(val status: Int, val responseString: String?) : StartChargeResult()
        class OnError(val errorCode: Int, val errorString: String?) : StartChargeResult()
    }

    suspend fun startCharge(timeout: Long): StartChargeResult = suspendCancellableCoroutine {
        val result = RobotApi.getInstance().startNaviToAutoChargeAction(0, timeout, object: ActionListener() {
            @Throws(RemoteException::class)
            override fun onResult(status: Int, responseString: String?, extraData: String? ) {
                log("startCharge::onResult($status, $responseString, $extraData)")
                it.resume(StartChargeResult.OnResult(status, responseString))
            }
            @Throws(RemoteException::class)
            override fun onError(errorCode: Int, errorString: String?, extraData: String?) {
                log("startCharge::onError($errorCode, $errorString, $extraData)")
                it.resume(StartChargeResult.OnError(errorCode, errorString))
            }
            @Throws(RemoteException::class)
            override fun onStatusUpdate(status: Int, data: String?, extraData: String?) {
                log("startCharge::onStatusUpdate($status, $data, $extraData)")
            }

        })
        log("startCharge::result($result)")
        if (result != Definition.RESULT_STOP)
            it.resume(StartChargeResult.Result(result))

    }
    fun stopCharge(): Int {
        return RobotApi.getInstance().stopAutoChargeAction(0, true)
    }



    sealed class LeaveChargePileResult {
        class Result(val result: Int) : LeaveChargePileResult()
        class OnResult(val result: Int, val message: String?) : LeaveChargePileResult()
    }

    suspend fun leaveChargePile(speed: Float, distance: Float): LeaveChargePileResult = suspendCancellableCoroutine {
        log("leaveChargePile($speed, $distance)")
        val result = RobotApi.getInstance().leaveChargingPile(0, speed, distance, object : CommandListener() {
            override fun onResult(result: Int, message: String?, extraData: String?) {
                log("leaveChargePile::onResult($result, $message, $extraData)")
                it.resume(LeaveChargePileResult.OnResult(result, message))
            }
            override fun onStatusUpdate(status: Int, data: String?, extraData: String?) {
                log("leaveChargePile::onStatusUpdate($status, $data, $extraData)")
            }
        })

        log("leaveChargePile::result($result)")
        if (result != Definition.RESULT_STOP)
            it.resume(LeaveChargePileResult.Result(result))
    }
    fun stopNavigation(): Int {
        return RobotApi.getInstance().stopNavigation(0)
    }

    val placeList: List<Pose>
        get() {
            return RobotApi.getInstance().placeList
        }
    fun isRobotExistLocations(destination: String): Boolean {
        if (destination.isEmpty())
            return false
        if (!isRobot)
            return true

        for (place in RobotApi.getInstance().placeList) {
            if (destination == place.name)
                return true
        }
        return false
    }
    fun isRobotInLocations(destination: String): Boolean {
        return if (isRobot)
            RobotApi.getInstance().isRobotInlocations(destination, 0.2)
        else
            true
    }

    sealed class StartNavigationResult {
        class Result(val result: Int) : StartNavigationResult()
        class OnResult(val status: Int, val responseString: String?) : StartNavigationResult()
        class OnError(val errorCode: Int, val errorString: String?) : StartNavigationResult()
    }
    suspend fun startNavigation(destination: String, time: Long, linearSpeed: Double, angularSpeed: Double): StartNavigationResult = suspendCancellableCoroutine {
        val result = RobotApi.getInstance().startNavigation(
            0,
            destination,
            0.2,
            time,
            linearSpeed,
            angularSpeed,
            object : ActionListener() {
                @Throws(RemoteException::class)
                override fun onResult(
                    status: Int,
                    responseString: String?,
                    extraData: String?
                ) {
                    log("startNavigation::onResult($status, $responseString)")
                    it.resume(StartNavigationResult.OnResult(status, responseString))
                }

                @Throws(RemoteException::class)
                override fun onError(errorCode: Int, errorString: String?, extraData: String?) {
                    log("startNavigation::onError($errorCode, $errorString)")
                    it.resume(StartNavigationResult.OnError(errorCode, errorString))
                }

                @Throws(RemoteException::class)
                override fun onStatusUpdate(status: Int, data: String?, extraData: String?) {
                    log("startNavigation::onStatusUpdate($status, $data)")
                }
            }
        )
        log("startNavigation::result($result)")
        if (result != Definition.RESULT_STOP)
            it.resume(StartNavigationResult.Result(result))
    }

    private fun log(text: String) {
        logMsf.update {
            text
        }
    }
}
